/* This file is part of the Palabos library.
 *
 * The Palabos softare is developed since 2011 by FlowKit-Numeca Group Sarl
 * (Switzerland) and the University of Geneva (Switzerland), which jointly
 * own the IP rights for most of the code base. Since October 2019, the
 * Palabos project is maintained by the University of Geneva and accepts
 * source code contributions from the community.
 *
 * Contact:
 * Jonas Latt
 * Computer Science Department
 * University of Geneva
 * 7 Route de Drize
 * 1227 Carouge, Switzerland
 * jonas.latt@unige.ch
 *
 * The most recent release of Palabos can be downloaded at
 * <https://palabos.unige.ch/>
 *
 * The library Palabos is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * The library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef PLB_COMPLEX_HH
#define PLB_COMPLEX_HH

#include <cmath>

#include "core/plbComplex.h"

namespace plb {

template <typename T>
T Complex<T>::pi = (T)4 * std::atan((T)1);

template <typename T>
Complex<T>::Complex() : Re(), Imag()
{ }

template <typename T>
Complex<T>::Complex(T Re_) : Re(Re_), Imag(T())
{ }

template <typename T>
Complex<T>::Complex(T Re_, T Imag_) : Re(Re_), Imag(Imag_)
{ }

template <typename T>
template <typename U>
Complex<T>::operator U() const
{
    return (U)Re;
}

template <typename T>
T Complex<T>::real() const
{
    return Re;
}

template <typename T>
T Complex<T>::imaginary() const
{
    return Imag;
}

template <typename T>
T Complex<T>::modulus() const
{
    return std::sqrt(sqrModulus());
}

template <typename T>
T Complex<T>::sqrModulus() const
{
    return Re * Re + Imag * Imag;
}

template <typename T>
Complex<T> Complex<T>::conjugate() const
{
    return Complex<T>(Re, -Imag);
}

template <typename T>
T Complex<T>::argument() const
{
    if (Re > T()) {
        return std::atan(Imag / Re);
    } else if (Re < T()) {
        return pi + std::atan(Imag / Re);
    } else {
        return pi / (T)2;
    }
}

template <typename T>
Complex<T> Complex<T>::intpow(int n) const
{
    T r_pow_n = std::pow((T)modulus(), (T)n);
    T phi = argument();
    return Complex<T>(r_pow_n * (std::cos(n * phi)), r_pow_n * (std::sin(n * phi)));
}

template <typename T>
Complex<T> &Complex<T>::operator+=(Complex<T> const &rhs)
{
    Re += rhs.Re;
    Imag += rhs.Imag;
    return *this;
}

template <typename T>
template <typename U>
Complex<T> &Complex<T>::operator+=(U rhs)
{
    Re += (T)rhs;
    return *this;
}

template <typename T>
Complex<T> &Complex<T>::operator-=(Complex<T> const &rhs)
{
    Re -= rhs.Re;
    Imag -= rhs.Imag;
    return *this;
}

template <typename T>
template <typename U>
Complex<T> &Complex<T>::operator-=(U rhs)
{
    Re -= (T)rhs;
    return *this;
}
template <typename T>
Complex<T> Complex<T>::operator-() const
{
    return Complex<T>(-Re, -Imag);
}

template <typename T>
Complex<T> &Complex<T>::operator*=(Complex<T> const &rhs)
{
    T tmpRe = Re * rhs.Re - Imag * rhs.Imag;
    Imag = Imag * rhs.Re + Re * rhs.Imag;
    Re = tmpRe;
    return *this;
}

template <typename T>
template <typename U>
Complex<T> &Complex<T>::operator*=(U rhs)
{
    Re *= (T)rhs;
    Imag *= (T)rhs;
    return *this;
}

template <typename T>
Complex<T> &Complex<T>::operator/=(Complex<T> const &rhs)
{
    T rhsNormSqr = rhs.sqrModulus();
    T tmpRe = (Re * rhs.Re + Imag * rhs.Imag) / rhsNormSqr;
    Imag = (Imag * rhs.Re - Re * rhs.Imag) / rhsNormSqr;
    Re = tmpRe;
    return *this;
}

template <typename T>
template <typename U>
Complex<T> &Complex<T>::operator/=(U rhs)
{
    Re /= (T)rhs;
    Imag /= (T)rhs;
    return *this;
}

template <typename T>
Complex<T> operator+(Complex<T> const &arg1, Complex<T> const &arg2)
{
    return Complex<T>(arg1) += arg2;
}

template <typename T, typename U>
Complex<T> operator+(Complex<T> const &arg1, U arg2)
{
    return Complex<T>(arg1) += (T)arg2;
}

template <typename T, typename U>
Complex<U> operator+(T arg1, Complex<U> const &arg2)
{
    return Complex<U>((U)arg1 + arg2.real(), (U)arg2.imaginary());
}

template <typename T>
Complex<T> operator-(Complex<T> const &arg1, Complex<T> const &arg2)
{
    return Complex<T>(arg1) -= arg2;
}

template <typename T, typename U>
Complex<T> operator-(Complex<T> const &arg1, U arg2)
{
    return Complex<T>(arg1) -= (T)arg2;
}

template <typename T, typename U>
Complex<U> operator-(T arg1, Complex<U> const &arg2)
{
    return Complex<T>((U)arg1 - arg2.real(), arg2.imaginary());
}

template <typename T>
Complex<T> operator*(Complex<T> const &arg1, Complex<T> const &arg2)
{
    return Complex<T>(arg1) *= arg2;
}

template <typename T, typename U>
Complex<T> operator*(Complex<T> const &arg1, U arg2)
{
    return Complex<T>(arg1) *= (T)arg2;
}

template <typename T, typename U>
Complex<U> operator*(T arg1, Complex<U> const &arg2)
{
    return Complex<U>((U)arg1 * arg2.real(), (U)arg1 * arg2.imaginary());
}

template <typename T>
Complex<T> operator/(Complex<T> const &arg1, Complex<T> const &arg2)
{
    return Complex<T>(arg1) /= arg2;
}

template <typename T, typename U>
Complex<T> operator/(Complex<T> const &arg1, U arg2)
{
    return Complex<T>(arg1) /= (T)arg2;
}

template <typename T, typename U>
Complex<U> operator/(T arg1, Complex<U> const &arg2)
{
    return Complex<U>((U)arg1 / arg2.real(), (U)arg1 / arg2.imaginary());
}

}  // namespace plb

#endif  // PLB_COMPLEX_HH
