/* This file is part of the Palabos library.
 *
 * The Palabos softare is developed since 2011 by FlowKit-Numeca Group Sarl
 * (Switzerland) and the University of Geneva (Switzerland), which jointly
 * own the IP rights for most of the code base. Since October 2019, the
 * Palabos project is maintained by the University of Geneva and accepts
 * source code contributions from the community.
 *
 * Contact:
 * Jonas Latt
 * Computer Science Department
 * University of Geneva
 * 7 Route de Drize
 * 1227 Carouge, Switzerland
 * jonas.latt@unige.ch
 *
 * The most recent release of Palabos can be downloaded at
 * <https://palabos.unige.ch/>
 *
 * The library Palabos is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * The library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/** \file
 * Flow in a lid-driven 2D cavity. The cavity is square and has no-slip walls,
 * except for the top wall which is driven to the right with a constant
 * velocity. The benchmark is challenging because of the velocity
 * discontinuities on corner nodes. The code on the other hand is very simple.
 * It could for example be used as a first example, to get familiar with Palabos.
 **/

#include <cmath>
#include <fstream>
#include <iomanip>
#include <iostream>
#include <vector>

#include "palabos2D.h"
#include "palabos2D.hh"  // include full template code

using namespace plb;
using namespace plb::descriptors;
using namespace std;

typedef double T;
#define DESCRIPTOR D2Q9Descriptor

void cavitySetup(
    MultiBlockLattice2D<T, DESCRIPTOR> &lattice, IncomprFlowParam<T> const &parameters,
    OnLatticeBoundaryCondition2D<T, DESCRIPTOR> &boundaryCondition)
{
    const plint nx = parameters.getNx();
    const plint ny = parameters.getNy();

    boundaryCondition.setVelocityConditionOnBlockBoundaries(lattice);

    setBoundaryVelocity(lattice, lattice.getBoundingBox(), Array<T, 2>((T)0., (T)0.));
    initializeAtEquilibrium(lattice, lattice.getBoundingBox(), (T)1., Array<T, 2>((T)0., (T)0.));

    T u = parameters.getLatticeU();
    setBoundaryVelocity(lattice, Box2D(1, nx - 2, ny - 1, ny - 1), Array<T, 2>(u, (T)0.));
    initializeAtEquilibrium(
        lattice, Box2D(1, nx - 2, ny - 1, ny - 1), (T)1., Array<T, 2>(u, (T)0.));

    lattice.initialize();
}

template <class BlockLatticeT>
void writeGif(BlockLatticeT &lattice, plint iter)
{
    const plint imSize = 600;

    ImageWriter<T> imageWriter("leeloo");
    imageWriter.writeScaledGif(
        createFileName("uNorm", iter, 6), *computeVelocityNorm(lattice), imSize, imSize);
    imageWriter.writeScaledGif(
        createFileName("logUnorm", iter, 6),
        *computeLog(*add((T)1.e-8, *computeVelocityNorm(lattice))), imSize, imSize);
}

template <class BlockLatticeT>
void writeVTK(BlockLatticeT &lattice, IncomprFlowParam<T> const &parameters, plint iter)
{
    T dx = parameters.getDeltaX();
    T dt = parameters.getDeltaT();
    VtkImageOutput2D<T> vtkOut(createFileName("vtk", iter, 6), dx);
    vtkOut.writeData<float>(*computeVelocityNorm(lattice), "velocityNorm", dx / dt);
    vtkOut.writeData<2, float>(*computeVelocity(lattice), "velocity", dx / dt);
}

int main(int argc, char *argv[])
{
    plbInit(&argc, &argv);

    global::directories().setOutputDir("./tmp/");

    IncomprFlowParam<T> parameters(
        (T)1e-2,  // uMax
        (T)100.,  // Re
        128,      // N
        1.,       // lx
        1.        // ly
    );
    const T logT = (T)0.1;
#ifndef PLB_REGRESSION
    const T imSave = (T)0.2;
    const T vtkSave = (T)1.;
    const T maxT = (T)10.1;
#else
    const T maxT = (T)0.51;
#endif

    writeLogFile(parameters, "2D cavity");

    MultiBlockLattice2D<T, DESCRIPTOR> lattice(
        parameters.getNx(), parameters.getNy(),
        new BGKdynamics<T, DESCRIPTOR>(parameters.getOmega()));

    OnLatticeBoundaryCondition2D<T, DESCRIPTOR> *
        // boundaryCondition = createLocalBoundaryCondition2D<T,DESCRIPTOR>();
        boundaryCondition = createInterpBoundaryCondition2D<T, DESCRIPTOR>();

    cavitySetup(lattice, parameters, *boundaryCondition);

#ifndef PLB_REGRESSION
    T previousIterationTime = T();
#endif

    // Main loop over time iterations.
    for (plint iT = 0; iT * parameters.getDeltaT() < maxT; ++iT) {
        global::timer("mainLoop").restart();

#ifndef PLB_REGRESSION
        if (iT % parameters.nStep(imSave) == 0 && iT > 0) {
            pcout << "Saving Gif ..." << endl;
            writeGif(lattice, iT);
            pcout << endl;
        }

        if (iT % parameters.nStep(vtkSave) == 0 && iT > 0) {
            pcout << "Saving VTK file ..." << endl;
            writeVTK(lattice, parameters, iT);
        }
#endif

        if (iT % parameters.nStep(logT) == 0) {
            pcout << "step " << iT << "; t=" << iT * parameters.getDeltaT();
        }

        // Lattice Boltzmann iteration step.
        lattice.collideAndStream();

        if (iT % parameters.nStep(logT) == 0) {
            pcout << "; av energy=" << setprecision(10) << getStoredAverageEnergy(lattice)
                  << "; av rho=" << getStoredAverageDensity(lattice) << endl;
#ifndef PLB_REGRESSION
            pcout << "Time spent during previous iteration: " << previousIterationTime << endl;
#endif
        }

#ifndef PLB_REGRESSION
        previousIterationTime = global::timer("mainLoop").stop();
#endif
    }

    delete boundaryCondition;
}
