/* This file is part of the Palabos library.
 *
 * The Palabos softare is developed since 2011 by FlowKit-Numeca Group Sarl
 * (Switzerland) and the University of Geneva (Switzerland), which jointly
 * own the IP rights for most of the code base. Since October 2019, the
 * Palabos project is maintained by the University of Geneva and accepts
 * source code contributions from the community.
 *
 * Contact:
 * Jonas Latt
 * Computer Science Department
 * University of Geneva
 * 7 Route de Drize
 * 1227 Carouge, Switzerland
 * jonas.latt@unige.ch
 *
 * The most recent release of Palabos can be downloaded at
 * <https://palabos.unige.ch/>
 *
 * The library Palabos is free software: you can redistribute it and/or
 * modify it under the terms of the GNU Affero General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * The library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Affero General Public License for more details.
 *
 * You should have received a copy of the GNU Affero General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <cmath>
#include <cstdlib>
#include <fstream>
#include <iostream>
#include <vector>

#include "palabos3D.h"
#include "palabos3D.hh"  // Use full generic version

using namespace plb;
using namespace std;

typedef double T;
#define DESCRIPTOR descriptors::D3Q19Descriptor

void cavitySetup(
    MultiBlockLattice3D<T, DESCRIPTOR> &lattice, IncomprFlowParam<T> const &parameters,
    OnLatticeBoundaryCondition3D<T, DESCRIPTOR> &boundaryCondition)
{
    const plint nx = parameters.getNx();
    const plint ny = parameters.getNy();
    const plint nz = parameters.getNz();
    Box3D topLid = Box3D(0, nx - 1, ny - 1, ny - 1, 0, nz - 1);
    Box3D everythingButTopLid = Box3D(0, nx - 1, 0, ny - 2, 0, nz - 1);

    boundaryCondition.setVelocityConditionOnBlockBoundaries(lattice);

    T u = std::sqrt((T)2) / (T)2 * parameters.getLatticeU();
    initializeAtEquilibrium(lattice, everythingButTopLid, (T)1., Array<T, 3>((T)0., (T)0., (T)0.));
    initializeAtEquilibrium(lattice, topLid, (T)1., Array<T, 3>(u, (T)0., u));
    setBoundaryVelocity(lattice, topLid, Array<T, 3>(u, (T)0., u));

    lattice.initialize();
}

void analyzeStrainRate(MultiBlockLattice3D<T, DESCRIPTOR> &lattice)
{
    Box3D slice(
        0, lattice.getNx() - 1, 0, lattice.getNy() - 1, lattice.getNz() / 2, lattice.getNz() / 2);
    std::unique_ptr<MultiTensorField3D<T, 6> > strainRate_LB(computeStrainRateFromStress(lattice));
    std::unique_ptr<MultiTensorField3D<T, 6> > strainRate_FD(
        computeStrainRate(*computeVelocity(lattice)));
    std::unique_ptr<MultiScalarField3D<T> > Snorm_LB(computeSymmetricTensorNorm(*strainRate_LB));
    std::unique_ptr<MultiScalarField3D<T> > Snorm_FD(computeSymmetricTensorNorm(*strainRate_FD));

#ifndef PLB_REGRESSION
    const plint imSize = 400;
    ImageWriter<T> imageWriter("leeloo");
    imageWriter.writeScaledGif("S_LB", *extractSubDomain(*Snorm_LB, slice), imSize, imSize);
    imageWriter.writeScaledGif("S_FD", *extractSubDomain(*Snorm_FD, slice), imSize, imSize);
    imageWriter.writeScaledGif(
        "S_LB_Trace", *extractSubDomain(*computeSymmetricTensorTrace(*strainRate_LB), slice),
        imSize, imSize);
    if (global::mpi().isMainProcessor()) {
        int err = 0;
        err = system("convert +append tmp/S_FD.gif tmp/S_LB.gif tmp/S_LB_Trace.gif tmp/strain.gif");
        if (err != 0)
            exit(err);
        err = system("/bin/rm tmp/S_FD.gif tmp/S_LB.gif tmp/S_LB_Trace.gif");
        if (err != 0)
            exit(err);
    }
#endif

    std::unique_ptr<MultiScalarField3D<T> > differenceSqr(
        computeSymmetricTensorNorm(*subtract(*strainRate_LB, *strainRate_FD)));

#ifndef PLB_REGRESSION
    imageWriter.writeScaledGif(
        "differenceSqr", *extractSubDomain(*differenceSqr, slice), imSize, imSize);
#endif

    T maxSnorm_LB = computeMax(*Snorm_LB);
    T maxSnorm_FD = computeMax(*Snorm_FD);
    pcout << "Max LB " << maxSnorm_LB << endl;
    pcout << "Max FD " << maxSnorm_FD << endl;
    T differenceMax = std::sqrt(computeMax(*differenceSqr));
    T differenceMin = std::sqrt(computeMin(*differenceSqr));

    pcout << "Absolute difference range: [" << differenceMin << "," << differenceMax << "]" << endl;
    pcout << "Relative difference range: [" << differenceMin / maxSnorm_LB << ","
          << differenceMax / maxSnorm_LB << "]" << endl;
}

int main(int argc, char *argv[])
{
    plbInit(&argc, &argv);
    global::directories().setOutputDir("./tmp/");

    IncomprFlowParam<T> parameters(
        (T)2e-2,  // uMax
        (T)40.,   // Re
        40,       // N
        1.,       // lx
        1.,       // ly
        1.        // lz
    );
    const T logT = (T)1 / (T)100;
    const T imSave = (T)0.2;
#ifndef PLB_REGRESSION
    const T maxT = (T)10.1;
#else
    const T maxT = 0.61;
#endif

    writeLogFile(parameters, "3D diagonal cavity");

    MultiBlockLattice3D<T, DESCRIPTOR> lattice(
        parameters.getNx(), parameters.getNy(), parameters.getNz(),
        new BGKdynamics<T, DESCRIPTOR>(parameters.getOmega()));

    OnLatticeBoundaryCondition3D<T, DESCRIPTOR> *boundaryCondition
        //= createInterpBoundaryCondition3D<T,DESCRIPTOR>();
        = createLocalBoundaryCondition3D<T, DESCRIPTOR>();

    cavitySetup(lattice, parameters, *boundaryCondition);

    // Main loop over time iterations.
    for (plint iT = 0; iT < parameters.nStep(maxT); ++iT) {
        if (iT % parameters.nStep(imSave) == 0) {
            pcout << "Processing analysis of the strain rate ..." << endl
                  << "==========================================" << endl
                  << endl;
            analyzeStrainRate(lattice);
        }

        if (iT % parameters.nStep(logT) == 0) {
            pcout << "Compute averages at time step " << iT << endl;
            pcout << "================================" << endl;
            pcout << "t=" << iT * parameters.getDeltaT() << endl;
            // Compute averages from the lattice. This takes a certain amount
            //   of time to be computed, because all lattice cells must be
            //   accessed.
            pcout << "Av energy=" << setprecision(10) << computeAverageEnergy(lattice)
                  << "; av rho=" << setprecision(10) << computeAverageDensity(lattice) << endl
                  << endl;
        }

        // Lattice Boltzmann iteration step.
        lattice.collideAndStream();

        if (iT % parameters.nStep(logT) == 0) {
            pcout << "Access internal statistics for time step " << iT << endl;
            pcout << "===========================================" << endl;
            // Display averages by accessing internal statistics of the lattice.
            //   This takes practically no time, because the averages where
            //   computed automatically during the lattice iteration cycle.
            pcout << "Av energy=" << setprecision(10) << getStoredAverageEnergy<T>(lattice)
                  << "; av rho=" << setprecision(10) << getStoredAverageDensity<T>(lattice) << endl
                  << endl;
        }
    }

    delete boundaryCondition;
}
